/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps, InspectorControls, InspectorAdvancedControls }                 from '@wordpress/block-editor';
import { Panel, PanelBody, PanelRow, SelectControl, TextControl, ToggleControl }       from "@wordpress/components";
import ServerSideRender                                                                from '@wordpress/server-side-render';

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function Edit( props ) {
	console.log("Hello World! (from profile-builder-profile-builder user listing block)");
	console.log(window.wppbUserListingBlockConfig);
	console.log(props.attributes);

	const blockProps = useBlockProps();

	return (
		<div { ...blockProps }>
			<InspectorControls>
				<Panel>
					<PanelBody title={ __( 'Listing Settings', 'profile-builder' ) } icon="more" initialOpen={true}>
						<PanelRow>
							<SelectControl
								label={ __( 'User Listing', 'profile-builder' ) }
								help={ __( 'Select the desired User Listing', 'profile-builder' ) }
								value={ props.attributes.name }
								options={ window.wppbUserListingBlockConfig.ul_name_options }
								onChange={(value) => {
									props.setAttributes({
															name: value,
														});
								}}
							/>
						</PanelRow>
						<PanelRow>
							<ToggleControl
								label={ __( 'Single', 'profile-builder' ) }
								help={ __( 'Select whether to show the Single User Listing template', 'profile-builder' ) }
								checked={ props.attributes.single }
								onChange={(value) => {
									props.setAttributes({
															single: value,
														});
								}}
							/>
						</PanelRow>
						{!props.attributes.single ?
							<PanelRow>
								<SelectControl
									label={ __( 'Meta Key', 'profile-builder' ) }
									help={ __( 'Select the Meta Name of a field. Only users that have the designated Meta Value for this field will be included in the Listing', 'profile-builder' ) }
									value={ props.attributes.meta_key }
									options={ window.wppbUserListingBlockConfig.meta_key_options }
									onChange={(value) => {
										props.setAttributes({
																meta_key: value,
															});
									}}
								/>
							</PanelRow>
							: ''}
						{!props.attributes.single && props.attributes.meta_key !== '' ?
							<PanelRow>
								<TextControl
									label={ __( 'Meta Value', 'profile-builder' ) }
									help={ __( 'Input the desired Meta Value for the selected field', 'profile-builder' ) }
									value={ props.attributes.meta_value }
									onChange={(value) => {
										props.setAttributes({
																meta_value: value,
															});
									}}
								/>
							</PanelRow>
							: ''}
						{!props.attributes.single ?
							<PanelRow>
								<TextControl
									label={ __( 'Include', 'profile-builder' ) }
									help={ __( 'Input a list of user IDs. Only the selected users will be included in the Listing', 'profile-builder' ) }
									value={ props.attributes.include_manual }
									onChange={(value) => {
										props.setAttributes({
																include_manual: value,
															});
									}}
								/>
							</PanelRow>
							: ''}
						{!props.attributes.single ?
							<PanelRow>
								<TextControl
									label={ __( 'Exclude', 'profile-builder' ) }
									help={ __( 'Input a list of user IDs. The selected users will be omitted from the Listing', 'profile-builder' ) }
									value={ props.attributes.exclude_manual }
									onChange={(value) => {
										props.setAttributes({
																exclude_manual: value,
															});
									}}
								/>
							</PanelRow>
							: ''}
						{props.attributes.single ?
							<PanelRow>
								<TextControl
									label={ __( 'ID', 'profile-builder' ) }
									help={ __( 'Input the ID for the desired user. Leaving this field blank will show the Single User Listing for the current user', 'profile-builder' ) }
									value={ props.attributes.id }
									onChange={(value) => {
										props.setAttributes({
																id: value,
															});
									}}
								/>
							</PanelRow>
							: ''}
					</PanelBody>
				</Panel>
			</InspectorControls>
			<InspectorAdvancedControls>
				<Panel>
					{!props.attributes.single ?
						<PanelRow>
							<TextControl
								label={ __( 'Meta Key', 'profile-builder' ) }
								help={ __( 'Manually type in the meta name of a field', 'profile-builder' ) }
								value={ props.attributes.meta_key }
								onChange={(value) => {
									props.setAttributes({
															meta_key: value,
														});
								}}
							/>
						</PanelRow>
						: ''}
				</Panel>
			</InspectorAdvancedControls>
			<ServerSideRender
				block="wppb/user-listing"
				attributes={ props.attributes }
			/>
		</div>
	);
}
